<?php

namespace Tests\Feature;

use App\User;
use App\Setting;
use Tests\TestCase;
use Illuminate\Support\Facades\Session;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Foundation\Testing\DatabaseTransactions;

class SettingTest extends TestCase
{
    // To reset DB to it's status before run test
    use DatabaseTransactions;

    // To create fake sentences to help us test and validation
    use WithFaker;

    private $user;


    protected function setUp() :void
    {
        parent::setUp();

        // to see description
        $this->withoutExceptionHandling();

        // Start session to enables csrf_token()
        Session::start();

        // Authenticate user
        $this->user = factory(User::class)->create(['role'=>'admin']);

        $this->actingAs($this->user);
    }

    /**
     * Method to ensure that the user can access to settings.
     *
     * @return void
     */
    public function testUserCanAccessSettingPage() :void
    {
        // setting gallery route which visit by user
        $response = $this->get(route('settings.index'));

        // Should be return status 200
        $response->assertStatus(200);
    }

    /**
     * Method to ensure that the user can read all settings.
     *
     * @return void
     */
    public function testUserCanReadUpdateSettings() :void
    {
        //When user visit the settings category
        $response = $this->get(route('settings.index'));

        // status should be 200
        $response->assertStatus(200);

        //He should be able to read the settings category
        $name = metaFields('settings', 'website_name', getCurrentLocale());
        $response->assertSee($name);
    }


    /**
     * Method to ensure that the update form route exists.
     *
     * @return void
     */

    public function testUserCanEditSetting() :void
    {
        //When user visit the settings category
        $response = $this->get(route('settings.index'));
        // status should be 200
        $response->assertStatus(200);

        $name = metaFields('settings', 'website_name', getCurrentLocale());

        //He should be able to read the settings category
        $response->assertSee($name);
    }

    /**
     * Method to ensure that the user can add settings.
     *
     * @return void
     */
    public function testUserCanAddSetting() :void
    {
        $dataToSave = $this->dataToSave();

        //When user submits post request to create video endpoint
        $response= $this->post(route('settings.store'), array_merge( $dataToSave, ['_token'=> csrf_token()]));

        // The redirect response header status is 302
        $response->assertStatus(302);

        // The response redirect to videos
        $response->assertRedirect('admin/settings');


        //He should be able to read the settings category
        foreach ($dataToSave as $key => $value) {
            $this->assertDatabaseHas('settings', [
                'key' => $key,
                'value' => $value,
            ]);
        }

    }

    /**
     * Method to ensure that the user can Edit settings.
     *
     * @return void
     */
    public function testUserCanUpdateSetting() :void
    {
        //Given we have settings in the database
        $dataToSave = $this->dataToSave();
        $dataToSave['update'] = true;

        //When user submits post request to create video endpoint
        $response= $this->post(route('settings.store'), array_merge($dataToSave, ['_token'=> csrf_token()]));


        // The redirect response header status is 302
        $response->assertStatus(302);

        // The response redirect to videos
        $response->assertRedirect('admin/settings');

        unset($dataToSave['update']);

        //He should be able to read the settings category
        foreach ($dataToSave as $key => $value) {
            $this->assertDatabaseHas('settings', [
                'key' => $key,
                'value' => $value,
            ]);
        }
    }

    /**
     * Data that send to save
     * @return array
     */
    public function dataToSave()
    {
        return [
            'website_name' => $this->faker->domainName,
            'keywords' => $this->faker->sentence,
            'meta_description' => $this->faker->sentence,
            'site_email' => $this->faker->companyEmail,
            'site_address' => $this->faker->sentence,
            'site_phone' => $this->faker->e164PhoneNumber,
            'mail_driver' => $this->faker->sentence,
            'mail_host' => $this->faker->sentence,
            'mail_port' => $this->faker->numberBetween(1111, 9999),
            'mail_username' => $this->faker->userName,
            'mail_password' => $this->faker->password,
            'mail_encryption' => $this->faker->word,
            'mail_from_address' =>$this->faker->companyEmail,
            'mail_from_name' => $this->faker->userName,
            'website_logo' => $this->faker->sentence,
            'website_disable' => 0,
            'dashboard_disable' => 0,
        ];
    }
}
