<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class SliderRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule( 'sliders' ) )
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules=[];

        foreach (languages() as $lang) {
            $lang_rules=[
                'first_title_'.$lang->local=> 'required|min:3|max:100',
                'second_title_'.$lang->local=> 'required|min:3|max:100',
//                'button_title_'.$lang->local=> 'required_with:button_url|nullable|min:3|max:50',
                'description_'.$lang->local=> 'required|min:3|max:400',
            ];

            $rules= array_merge($rules, $lang_rules);
        }

        $rules['image'] = 'required';

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('sliders');
        $local = getCurrentLocale();
        $attributes=[
            'image' => $field[$local]['image'] ?? __('sliders.image'),
            'button_url' => $field[$local]['button_url'] ?? __('sliders.button_url'),
            ];
        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? ' (عربي) ' : ' (English) ';

            $lang_attributes=[
                'title_'.$lang->local=> __('sliders.title') . $local,
                'first_title_'.$lang->local=> ($field[$lang->local]['first_title'] ?? __('sliders.first_title')) . $local,
                'second_title_'.$lang->local=> ($field[$lang->local]['second_title'] ?? __('sliders.second_title')) . $local,
                'button_title_'.$lang->local=> ($field[$lang->local]['button_title'] ?? __('sliders.button_title')) . $local,
                'description_'.$lang->local=> ($field[$lang->local]['description'] ?? __('sliders.description')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }

    public function messages()
    {
        $message = [];
        foreach (languages() as $lang) {
            $lang_messages=[
                'button_title_'.$lang->local.'.required_with'=> __('sliders.button_title_required_if'),
            ];
            $message= array_merge($message, $lang_messages);
        }
        return $message;
    }

    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            parent::failedValidation($validator);
        }
    }
}
