<?php

namespace App\Http\Requests;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Request;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

/**
 * @property mixed video_type
 * @property mixed video_url
 * @property mixed images
 */
class ArticleRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        if (! Auth::check()) {
            return false;
        }
        if(in_array(Auth::user()->role, ['super_admin', 'admin', 'sub_admin']))
        {
            return true;
        }

        if(Auth::user()->hasModule('articles'))
        {
            return true;
        }
        return Auth::check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $rules = [
            'news_date' => 'required|date'
        ];

        foreach (languages() as $lang) {
            $lang_rules=[
                'title_'.$lang->local=> 'required|min:3|max:100',
                'slug_'.$lang->local=> [
                    "required",
                    "min:3",
                    "max:100",
                    Rule::unique('article_descriptions', 'slug')->ignore($this->article? $this->article->id : 0, 'article_id')
                ],
                'keywords_'.$lang->local=> 'required|min:3|max:400',
                'meta_description_'.$lang->local=> 'required|min:3|max:400',
                'description_'.$lang->local=> 'required|min:3',
            ];
            $rules= array_merge($rules, $lang_rules);
        }

        $rules['image'] = 'required';

        return $rules;
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('articles');
        $local = getCurrentLocale();
        $attributes=[
            'news_date' =>  $field[$local]['news_date'] ?? __('articles.news_date'),
            'image' =>  $field[$local]['image'] ?? __('articles.image'),
        ];


        foreach (languages() as $lang) {
            $local = $lang->local == 'ar' ? ' (عربي) ' : ' (English) ';
            $lang_attributes=[
                'title_'.$lang->local => ($field[$lang->local]['title'] ?? __('articles.title')) . $local,
                'description_'.$lang->local=> ($field[$lang->local]['description'] ?? __('articles.description')) . $local,
                'slug_'.$lang->local=> ($field[$lang->local]['slug'] ?? __('articles.slug')) . $local,
                'meta_description_'.$lang->local=> ($field[$lang->local]['meta_description'] ?? __('articles.meta_description')) . $local,
                'keywords_'.$lang->local=> ($field[$lang->local]['keywords'] ?? __('articles.keywords')) . $local,
            ];
            $attributes= array_merge($attributes, $lang_attributes);
        }
        return $attributes;
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator)
    {
        // Change response attitude if the request done via Ajax requests like API requests
        if(Request::wantsJson()){
            $errors = (new ValidationException($validator))->errors();
            throw new HttpResponseException(response()->json(['errors' => $errors], 422));
        } elseif (Request::ajax()){
            $errors = $validator->errors()->first();
            throw new HttpResponseException(response()->json(['errors' => $errors]));
        } else {
            parent::failedValidation($validator);
        }
    }
}
