<?php

namespace App\Http\Requests\API;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class AddBookingRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     * So the user must be guest
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $phonePattern = '/^(5){1}([0-9]{8})$/';
        $codePattern = '/^[0-9]{4}$/';
        $namesRegex = '/^([a-zA-Z ]|[\p{Arabic}])*$/u';
        return [
            'branch_id' => 'required|exists:branches,id',
            'specific_doctor_way' => 'required|in:1,2',
            'doctor_id' => 'required_if:specific_doctor_way,1|exists:doctors,id',
            'appointment_date' => 'required|exists:appointments,appointment_date',
            'appointment_time' => 'required|exists:appointment_times,id',
            'name' => ['required','min:3','max:100','regex:'.$namesRegex],
            'email' => 'required|email|max:50',
            'phone' => ['required','regex:'.$phonePattern],
            'verify_code' => ['required', 'regex:'.$codePattern],
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        $field = metaFields('reservations');
        $local = getCurrentLocale();
        $attributes=[
            'branch_id' => $field[$local]['branch_id'] ?? __('reservations.branch_id'),
            'specific_doctor_way' => $field[$local]['specific_doctor_way'] ?? __('reservations.specific_doctor_way'),
            'doctor_id' => $field[$local]['doctor_id'] ?? __('reservations.doctor_id'),
            'appointment_date' => $field[$local]['appointment_date'] ?? __('reservations.appointment_date'),
            'appointment_time' => $field[$local]['appointment_time'] ?? __('reservations.appointment_time'),
            'name' => $field[$local]['name'] ?? __('reservations.name'),
            'phone' => $field[$local]['phone'] ?? __('reservations.phone'),
            'email' => $field[$local]['email'] ?? __('reservations.email'),
            'verify_code' => $field[$local]['verify_code'] ?? __('members.verify_code'),

        ];
        return $attributes;
    }

    public function messages()
    {
        return [
            'doctor_id.required_if' => __('reservations.doctor_id_validation'),
        ];
    }


    /**
     * Handle a failed validation attempt.
     *
     * Override the parent method action if the request is for AJAX
     *
     * @param Validator $validator
     * @return void
     *
     * @throws ValidationException
     */
    protected function failedValidation(Validator $validator) :void
    {
        // Change response attitude if the request done via Ajax requests like API requests
        $errors = (new ValidationException($validator))->errors();
        $message = $validator->errors()->first();
        throw new HttpResponseException(response()->json(['message'=> $message, 'errors' => $errors],  400));
    }
}
